/****************************************************************************
 * Copyright (c) 2007 Einir Valdimarsson and Chrysanthe Preza
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 ****************************************************************************/

#include "estimateDVCG.h"
#include "arrayManip.h"
#include "RectDomainIter.h"
#include "blitz/fftwInterface.h"
#include "estimateCGMLpoisson.h"

namespace cosm {

// class constructor
template<typename T, int N>
EstimateDVCG<T,N>::EstimateDVCG(
    Array< Array<T,N>, 1>& psfs,
    Array<T,N>& img,
    int iterations,
    T regularization,
    T epsilon,
    bool adjoint
) : 
    EstimateIterative<T,N>(psfs(0), img, iterations),
    regularization_(regularization), epsilon_(epsilon), adjoint_(adjoint)
{

    TinyVector<int,N> sizeImg = this->img_.length();
    TinyVector<int,N> sizePsf = this->psf_.length();

    // Read multiple PSFs
    psfs_.resize(psfs.length(0));
    for ( int m = 0; m < psfs.length(0); m++ ) 
    {
        psfs_(m).resize(sizeImg+sizePsf);
        padCenter(psfs(m), psfs_(m));
    }

    //Init
    s_.resize(this->img_.extent());
    s_ = 0;

    dk_.resize(this->img_.extent());
    dk_ = 0;

    Kss_.resize(this->img_.extent());
    Kss_ = 0;

    rk_.resize(this->img_.extent());
    rk_ = 0;

    b_.resize(this->img_.extent());
    b_ = 0;

    rkNormOld_ = 1;
    nrk_ = 0;
    gamma_ = 0;
    lambda_ = 0;
    beta_ = regularization_;

    //Pre-process
    Array<T, N> f_;
    f_.resize(this->img_.extent());
    f_ = this->img_;

    s_ = where ( f_<= 0, sqrt(epsilon_), sqrt(f_));

    Array<T, N> tmp;
    tmp.resize(s_.extent());
    tmp = pow2(s_);

    dvftconv<T, N>(tmp, psfs_, Kss_, adjoint_);
}

// Function to specify the psf for new estimation
template<typename T, int N>
void EstimateDVCG<T,N>::setPSF(
    Array<T,N>& psf
) {
    // not implemented
}

// Function to specify the image for new estimation
// NOTE: new img has to be the same size as previous img
template<typename T, int N>
void EstimateDVCG<T,N>::setImage(
    Array<T,N>& img
) {
    EstimateIterative<T,N>::setImage(img);
}

// Function for single iteration of the em algorithm
template<typename T, int N>
void EstimateDVCG<T,N>::iterate(
){
    Grad<T, N>(Kss_, s_, this->img_, b_, beta_, psfs_, rk_);

    nrk_ = sum(pow2(rk_));

    if (fmod(double(this->iterationsDone_), 5) == double(0)){
        dk_ = rk_;    //initialize / re`start CG
    }else{
        gamma_ = nrk_/rkNormOld_;    //new conjugate direction
        dk_ = rk_ + gamma_ * dk_;     //Fletcher Reeves
    }

    rkNormOld_ = nrk_;

    lambda_ = NewtonRaphson<T, N>(Kss_, this->img_, b_, s_, dk_, beta_, psfs_, Kss_, adjoint_);

    s_ = s_ + lambda_ * dk_;

    this->est_ = pow2(s_);

}

}
